--------------------------------------------------------------------------------
-- Konfiguration des foreign server (Verbindungsdaten)
CREATE OR REPLACE FUNCTION tsystem.fdw__get_server_options(
  IN  server_name varchar,
  OUT options     varchar,
  OUT srvfdw      integer
) RETURNS record AS $$
DECLARE
  _options varchar;
BEGIN
  WITH
    serveroptions AS
    (
      SELECT pg_catalog.pg_foreign_server.srvfdw, unnest( srvoptions ) AS option FROM pg_catalog.pg_foreign_server WHERE srvname = server_name
    ),
    serveroptions_sorted AS
    (
      SELECT
        serveroptions.srvfdw,
        serveroptions.option
      FROM
        serveroptions
      ORDER BY
        option
    )
  SELECT
    string_agg( serveroptions_sorted.option, ' ' ),
    serveroptions_sorted.srvfdw
  FROM
    serveroptions_sorted
  GROUP BY
    serveroptions_sorted.srvfdw
  INTO
    options,
    srvfdw;
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- Überprüft ob ein foreign server mit exakt diesen Verbindungsdaten existiert
CREATE OR REPLACE FUNCTION tsystem.fdw__get_server_doesnotexistsorsame(
  extension_name    varchar,
  server_name       varchar,
  fdw_host          varchar,
  fdw_port          varchar,
  fdw_database      varchar
) RETURNS varchar AS $$
DECLARE
  _options   varchar;
  _srvfdw    integer;
  _extension integer;
  _compare   varchar;
BEGIN
  SELECT
    options,
    srvfdw
  FROM
    tsystem.fdw__get_server_options(server_name)
  INTO
    _options,
    _srvfdw;

  IF _options IS NULL THEN
    RETURN false;
  END IF;

  SELECT
    oid
  FROM
    pg_catalog.pg_foreign_data_wrapper
  WHERE
    fdwname = extension_name
  INTO
    _extension;

  _compare := CONCAT_WS(
    ' ',
    CONCAT_WS( '=', 'dbname', fdw_database ),
    CONCAT_WS( '=', 'host',   fdw_host ),
    CONCAT_WS( '=', 'port',   fdw_port )
  );

  RETURN ( ( _options = _compare )  AND (_srvfdw = _extension) );
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- foreign server erstellen oder öndern (overwrite = true)
-- Schlägt fehl, wenn bereits ein foreign server mit anderer Konfiguration existiert
CREATE OR REPLACE FUNCTION tsystem.fdw__create_server(
  extension_name    varchar,
  server_name       varchar,
  fdw_host          varchar,
  fdw_port          varchar,
  fdw_database      varchar,
  overwrite         boolean
) RETURNS boolean AS $$
DECLARE
  _options_old varchar;
  _options_new varchar;
  _exists      boolean;
  _cmd         varchar;
BEGIN
  -- fdw extension erstellen --
  IF NOT EXISTS (SELECT true FROM pg_extension WHERE extname = extension_name ) THEN
    EXECUTE 'CREATE EXTENSION IF NOT EXISTS ' || extension_name || ';';
  ELSE
    RAISE NOTICE 'Extension % existiert bereits', extension_name;
  END IF;

  -- Server existiert bereits genau so --
  _exists := tsystem.fdw__get_server_doesnotexistsorsame(
    extension_name ,
    server_name    ,
    fdw_host       ,
    fdw_port       ,
    fdw_database
  );
  IF _exists THEN
    RAISE NOTICE 'FDW Server % existiert bereits [%, %, %, %]', server_name, extension_name, fdw_host, fdw_port, fdw_database;
    RETURN true;
  END IF;

  -- Server existiert bereits mit anderen Optionen --
  _options_old := (tsystem.fdw__get_server_options(server_name)).options;
  IF NOT _options_old IS NULL THEN
    RAISE NOTICE 'FDW Server % existiert bereits [%]', server_name, _options_old;
    IF NOT overwrite THEN
      RETURN false;
    END IF;

    _options_new := 'SET host ' || quote_literal(fdw_host) || ', SET port ' || quote_literal(fdw_port) || ', SET dbname ' || quote_literal(fdw_database);
    RAISE NOTICE 'FDW Server % wird geändert [%]', server_name, _options_new;
    EXECUTE 'ALTER SERVER ' || server_name || '
               OPTIONS ( ' || _options_new || ');';

    RETURN true;
  END IF;

  -- Server mit angegebenen Optionen neu erstellen --
  _options_new := 'host ' || quote_literal(fdw_host) || ', port ' || quote_literal(fdw_port) || ', dbname ' || quote_literal(fdw_database);
  RAISE NOTICE 'Erstelle fdw server % [%]', server_name, _options_new;
  _cmd := 'CREATE SERVER ' || server_name || '
             FOREIGN DATA WRAPPER ' || extension_name || '
             OPTIONS (' || _options_new || ');';
  EXECUTE _cmd;
  RETURN true;
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- user mapping für eine lokale Rolle für einen foreign server erstellen oder ändern
CREATE OR REPLACE FUNCTION tsystem.fdw__createorreplace_user_mapping(
  server_name  varchar,
  user_name    varchar,
  fdw_username varchar,
  fdw_password varchar DEFAULT ''
) RETURNS void
  AS $$
DECLARE
  _exists  boolean;
  _options varchar;
BEGIN
  BEGIN
    SELECT
      true
    FROM
      pg_catalog.pg_roles
    WHERE
      rolname = user_name
    INTO
      _exists;

    IF ( ( _exists IS NULL ) OR ( NOT _exists ) ) THEN
      RAISE NOTICE 'Rolle % exisitiert nicht. Unmöglich user mapping zu erstellen.', user_name;
    RETURN;
    END IF;

    SELECT
      true
    FROM
      pg_user_mappings
    WHERE
      srvname = server_name
      AND usename = user_name
    INTO
      _exists;

    IF _exists THEN
      _options := 'SET "user" ' || quote_literal( fdw_username ) || ', SET password ' || quote_literal( fdw_password );
      RAISE NOTICE 'Ändere user mapping für %@% [%]', user_name, server_name, _options;
      EXECUTE 'ALTER USER MAPPING FOR "' || user_name || '"
                 SERVER ' || server_name || '
                 OPTIONS (' || _options || ')';
    ELSE
      _options := '"user" ' || quote_literal( fdw_username ) || ', password ' || quote_literal( fdw_password );
      RAISE NOTICE 'Erstelle user mapping für %@% [%]', user_name, server_name, _options;
      EXECUTE 'CREATE USER MAPPING FOR "' || user_name || '"
                 SERVER ' || server_name || '
                 OPTIONS (' || _options || ')';
    END IF;
  EXCEPTION
    WHEN OTHERS THEN
      RAISE NOTICE 'ERROR (fdw__createorreplace_user_mapping): %', sqlerrm;
  END;
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- Erstellt für alle LLV Einträge, welche eine Login-Rolle besitzen ein user mapping
CREATE OR REPLACE FUNCTION tsystem.fdw__createorreplace_user_mappings_llv(
  server_name varchar
) RETURNS void
  AS $$
DECLARE
  _rec record;
BEGIN
  FOR _rec IN SELECT trim( ll_ad_krz ) AS ll_ad_krz FROM llv LOOP
    PERFORM tsystem.fdw__createorreplace_user_mapping( server_name, _rec.ll_ad_krz, _rec.ll_ad_krz );
  END LOOP;
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- Erstellt eine DB-Funktion die als dbrid DEFAULT in einer foreign table benutzt
-- werden kann. Benutzt dazu dblink (und die hinterlegte Konfiguration für den foreign server)
-- um auf die dbrid Sequenz im foreign server zuugreifen
CREATE OR REPLACE FUNCTION tsystem.fdw__createorreplace_db_id_seq(
  server_name varchar
  ) RETURNS void
  AS $$
DECLARE
  _cmd varchar;
BEGIN
  _cmd := 'CREATE OR REPLACE FUNCTION tsystem.fdw__' || server_name || '_db_id_seq() RETURNS character varying AS $inner$
DECLARE
  _connstring varchar;
BEGIN
  _connstring := (tsystem.fdw__get_server_options(''' || server_name || ''')).options || '' user='' || current_user;
  RETURN * FROM dblink( _connstring, ''SELECT nextval(''''db_id_seq''''::regclass) AS result;'' ) AS temp ( result integer );
END $inner$ language plpgsql';
  EXECUTE _cmd;
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- Erstellt eine foreign table in einem foreign server
CREATE OR REPLACE FUNCTION tsystem.fdw__create_table(
  server_name varchar,
  schema_name varchar,
  fdw_schema  varchar,
  fdw_table   varchar
  ) RETURNS void
  AS $$
BEGIN
  EXECUTE 'CREATE SCHEMA IF NOT EXISTS ' || schema_name || ';';
  EXECUTE 'IMPORT FOREIGN SCHEMA ' || fdw_schema || ' LIMIT TO (' || fdw_table || ')
           FROM SERVER ' || server_name || ' INTO ' || schema_name || ';';
END  $$ language plpgsql;

--------------------------------------------------------------------------------
-- Erstellt foreign server, user mapping für 'postgres' Benutzer und foreign table
-- with_usermappings_llv = true -> Aufruf tsystem.fdw__createorreplace_user_mappings_llv
CREATE OR REPLACE FUNCTION tsystem.fdw__create(
  extension_name        varchar,
  server_name           varchar,
  schema_name           varchar,
  fdw_host              varchar,
  fdw_port              varchar,
  fdw_database          varchar,
  fdw_schema            varchar,
  fdw_table             varchar,
  with_usermappings_llv boolean
) RETURNS void
  AS $$
DECLARE
  rec record;
  _ret boolean;
BEGIN
  -- Server erstellen
  EXECUTE 'SELECT
    tsystem.fdw__create_server(
    $1,
    $2,
    $3,
    $4,
    $5,
    $6
  )'
  USING
    extension_name,
    server_name,
    fdw_host,
    fdw_port,
    fdw_database,
    false
  INTO
    _ret;

  PERFORM tsystem.fdw__createorreplace_user_mapping(_server_name, 'postgres', 'postgres');

  IF (_ret AND with_usermappings) THEN
    -- User Mapppings
    PERFORM tsystem.fdw__createorreplace_user_mappings_llv( server_name );
  END IF;

  -- Tabelle
  PERFORM tsystem.fdw__create_table( server_name, schema_name, fdw_schema, fdw_table );
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- Benutzerfreundliche Funktion ohne jegliche Parameter um fdw text0 zu erstellen
CREATE OR REPLACE FUNCTION tsystem.fdw__centralizedtext0_create(
  ) RETURNS void
AS $$
DECLARE
  _extension_name        varchar;
  _server_name           varchar;
  _schema_name           varchar;
  _fdw_host              varchar;
  _fdw_port              varchar;
  _fdw_database          varchar;
  _fdw_schema            varchar;
  _fdw_table             varchar;
  _overwrite             boolean;
  _ret                   boolean;
  _exists                boolean;
BEGIN
  _extension_name        := 'postgres_fdw';
  _server_name           := 'zentraletext0';
  _schema_name           := 'public';
  _fdw_host              := 'localhost';
  _fdw_port              := '5434';
  _fdw_database          := 'PRODAT-DEMO-21.11';
  _fdw_schema            := 'public';
  _fdw_table             := 'text0';
  _overwrite             := true;


  EXECUTE 'SELECT
    tsystem.fdw__create_server(
    $1,
    $2,
    $3,
    $4,
    $5,
    $6
)'
  USING
    _extension_name,
    _server_name,
    _fdw_host,
    _fdw_port,
    _fdw_database,
    _overwrite
  INTO
    _ret;

  IF _ret THEN
    -- lokale Tabelle umbenennen
    -- Nur wenn es auch eine 'echte' Tabelle ist
    SELECT
      true
    FROM
      pg_catalog.pg_tables
    WHERE
      tablename = 'text0'
    INTO
      _exists;

    IF _exists THEN
      -- 'echte' lokale Tabelle sichern
      EXECUTE 'ALTER TABLE IF EXISTS public.text0 RENAME TO text0_local_fdw;';
    ELSE
      -- existierende FDW Tabelle löschen
      -- explizit textx löschen, so dass knallt, sollten in Zukunft weitere
      -- abhängige DB-Objekte erstellet werden
      EXECUTE 'DROP VIEW IF EXISTS textx;';
      EXECUTE 'DROP FOREIGN TABLE IF EXISTS text0;';
    END IF;

    -- User Mappings erstellen
    PERFORM tsystem.fdw__createorreplace_user_mapping(_server_name, 'postgres', 'postgres');
    PERFORM tsystem.fdw__createorreplace_user_mappings_llv( _server_name );
    -- Tabelle erstellen
    PERFORM tsystem.fdw__create_table( _server_name, _schema_name, _fdw_schema, _fdw_table );

    -- Funktion um externe Sequenz benutzen zu können
    EXECUTE 'SELECT tsystem.fdw__createorreplace_db_id_seq( $1 )' USING _server_name;
    -- Tabelle umschreiben um externe Sequenz zu benutzen
    ALTER FOREIGN TABLE text0 ALTER COLUMN dbrid SET DEFAULT tsystem.fdw__zentraletext0_db_id_seq();

    -- VIEW wiederherstellen
    CREATE OR REPLACE VIEW textx AS
      SELECT (
        CASE
          WHEN prodat_languages.curr_lang()='CH' THEN COALESCE(t_kundtxt0, t_feld0)
          WHEN prodat_languages.curr_lang()='F'  THEN COALESCE(t_kundtxt1, t_feld1)
          WHEN prodat_languages.curr_lang()='D'  THEN COALESCE(t_kundtxt0, t_feld0)
          WHEN prodat_languages.curr_lang()='EN' THEN COALESCE(t_kundtxt3, t_feld3)
        ELSE
          t_feld0
        END
      ) AS t_feld, t_nr FROM text0;
  END IF;

  RETURN;
END $$ language plpgsql;

--------------------------------------------------------------------------------
-- Benutzerfreundliche Funktion ohne jegliche Parameter um fdw text0 zu entfernen
CREATE OR REPLACE FUNCTION tsystem.fdw__centralizedtext0_remove(
  ) RETURNS void
  AS $$
DECLARE
  _exists boolean;
BEGIN
  -- lokale Tabelle wiederherstellen

  -- Nur wenn es auch eine foreign Tabelle gibt
  SELECT
    true
  FROM
    pg_class
  INNER JOIN
    pg_foreign_table ON pg_class.oid = pg_foreign_table.ftrelid
  WHERE
    pg_class.relname = 'text0'
  INTO
    _exists;

  IF _exists THEN
    -- Nur wenn es auch eine lokale Tabelle gibt
    SELECT
      true
    FROM
      pg_catalog.pg_tables
    WHERE
      tablename = 'text0_local_fdw'
    INTO
      _exists;

    IF _exists THEN
      -- existierende FDW Tabelle löschen
      -- explizit textx löschen, so dass knallt, sollten in Zukunft weitere
      -- abhängige DB-Objekte erstellet werden
      EXECUTE 'DROP VIEW IF EXISTS textx;';
      EXECUTE 'DROP FOREIGN TABLE IF EXISTS public.text0;';
      EXECUTE 'ALTER TABLE IF EXISTS public.text0_local_fdw RENAME TO text0;';

      -- VIEW wiederherstellen
      CREATE OR REPLACE VIEW textx AS
      SELECT (
        CASE
          WHEN prodat_languages.curr_lang()='CH' THEN COALESCE(t_kundtxt0, t_feld0)
          WHEN prodat_languages.curr_lang()='F'  THEN COALESCE(t_kundtxt1, t_feld1)
          WHEN prodat_languages.curr_lang()='D'  THEN COALESCE(t_kundtxt0, t_feld0)
          WHEN prodat_languages.curr_lang()='EN' THEN COALESCE(t_kundtxt3, t_feld3)
        ELSE
          t_feld0
        END
      ) AS t_feld, t_nr FROM text0;
    ELSE
      RAISE NOTICE 'Keine lokale umbenannte text0 gefunden';
    END IF;
  ELSE
    RAISE NOTICE 'Keine FDW Tabelle für text0 gefunden';
  END IF;
END $$ language plpgsql;